<?php
namespace SabaiApps\Directories\Component\Social;

use SabaiApps\Directories\Component\AbstractComponent;
use SabaiApps\Directories\Component\Field;

class SocialComponent extends AbstractComponent implements
    IMedias,
    Field\ITypes,
    Field\IWidgets,
    Field\IFilters,
    Field\IRenderers
{
    const VERSION = '1.14.38', PACKAGE = 'directories';
    
    public static function description()
    {
        return 'Enables social media account fields and display content fetched from social media sites.';
    }
    
    public function socialMediaNames()
    {
        $names = ['facebook', 'twitter', 'pinterest', 'tumblr', 'linkedin', 'flickr', 'youtube', 'instagram', 'rss', 'github', 'telegram', 'tiktok', 'threads', 'bluesky', 'mastodon'];
        if ($custom = $this->_application->Filter('social_medias', [])) {
            $names = array_merge($names, array_keys($custom));
        }
        return $names;
    }
    
    public function socialMediaInfo($name)
    {
        switch ($name) {
            case 'facebook': 
                return [
                    'type' => 'textfield',
                    'label' => 'Facebook',
                    'icon' => 'fab fa-facebook-square',
                    //'regex' => '/^https?:\/\/((w{3}\.)?)facebook.com\/.*/i',
                    'default' => 'facebook',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'Facebook'),
                ];
            case 'twitter': 
                return [
                    'type' => 'textfield',
                    'label' => 'X (Twitter)',
                    'icon' => 'fab fa-square-x-twitter',
                    //'regex' => '/^https?:\/\/twitter\.com\/(#!\/)?[a-z0-9_]+[\/]?$/i',
                    'default' => 'twitter',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'X (Twitter)'),
                ];
            case 'pinterest': 
                return [
                    'type' => 'textfield',
                    'label' => 'Pinterest',
                    'icon' => 'fab fa-pinterest-square',
                    'default' => 'pinterest',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'Pinterest'),
                ];
            case 'instagram': 
                return [
                    'type' => 'textfield',
                    'label' => 'Instagram',
                    'icon' => 'fab fa-instagram-square',
                    'default' => 'instagram',
                    'placeholder' => __('Enter Instagram username or URL. Prefix with "#" if hashtag.', 'directories'),
                ];
            case 'youtube': 
                return [
                    'type' => 'textfield',
                    'label' => 'YouTube',
                    'icon' => 'fab fa-youtube-square',
                    'default' => 'YouTube',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'YouTube'),
                ];
            case 'tumblr': 
                return [
                    'label' => 'Tumblr',
                    'icon' => 'fab fa-tumblr-square',
                    'default' => 'http://staff.tumblr.com/',
                ];
            case 'linkedin': 
                return array(
                    'label' => 'LinkedIn',
                    'icon' => 'fab fa-linkedin',
                    'default' => 'https://www.linkedin.com/company/linkedin',
                );
            case 'flickr': 
                return [
                    'label' => 'Flickr',
                    'icon' => 'fab fa-flickr',
                    'default' => 'https://www.flickr.com/people/flickr',
                ];
            case 'rss': 
                return [
                    'label' => 'RSS',
                    'icon' => 'fas fa-rss-square',
                    'default' => $this->_application->getPlatform()->getSiteUrl(),
                ];
            case 'github':
                return [
                    'type' => 'textfield',
                    'label' => 'GitHub',
                    'icon' => 'fab fa-github-square',
                    'default' => 'github',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'GitHub'),
                ];
            case 'telegram':
                return [
                    'type' => 'textfield',
                    'label' => 'Telegram',
                    'icon' => 'fab fa-telegram',
                    'default' => 'telegram',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'Telegram'),
                ];
            case 'tiktok':
                return [
                    'type' => 'textfield',
                    'label' => 'TikTok',
                    'icon' => 'fab fa-tiktok',
                    'default' => '@tiktok',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'TikTok'),
                ];
            case 'threads':
                return [
                    'type' => 'textfield',
                    'label' => 'Threads',
                    'icon' => 'fab fa-square-threads',
                    'default' => '@threads',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'Threads'),
                ]; 
            case 'bluesky':
                return [
                    'type' => 'textfield',
                    'label' => 'Bluesky',
                    'icon' => 'fab fa-square-bluesky',
                    'default' => '@bsky.app',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'Bluesky'),
                ]; 
            case 'mastodon':
                return [
                    'type' => 'textfield',
                    'label' => 'Mastodon',
                    'icon' => 'fab fa-mastodon',
                    'default' => '@tiktok',
                    'placeholder' => sprintf(__('Enter %s username or URL.', 'directories'), 'Mastodon'),
                ];                
            default:
                $custom = $this->_application->Filter('social_medias', []);
                if (isset($custom[$name])) {
                    return $custom[$name];
                }
        }
    }

    public function socialMediaUrl($name, $value)
    {
        if (strpos($value, 'https://') === 0
            || strpos($value, 'http://') === 0
        ) return $value;

        switch ($name) {
            case 'facebook':
                if (strpos($value, '!') === 0) {
                    $value = substr($value, 1);
                }
                return 'https://www.facebook.com/' . rawurlencode($this->_removeUrlPrefix($value, 'facebook.com'));
            case 'twitter':
                if (strpos($value, '#') === 0) {
                    return 'https://x.com/hashtag/' . rawurlencode(substr($value, 1));
                }
                if (strpos($value, '@') === 0) {
                    $value = rawurlencode(substr($value, 1));
                }
                return 'https://x.com/' . rawurlencode($this->_removeUrlPrefix($value, strpos($value, 'twitter.com') !== false ? 'twitter.com' : 'x.com'));
            case 'pinterest':
                return 'https://www.pinterest.com/' . rawurlencode($value);
            case 'instagram':
                if (strpos($value, '#') === 0) {
                    return 'https://instagram.com/explore/tags/' . rawurlencode(substr($value, 1));
                }
                return 'https://instagram.com/' . rawurlencode($this->_removeUrlPrefix($value, 'instagram.com'));
            case 'youtube':
                return 'https://www.youtube.com/@' . rawurlencode(ltrim($this->_removeUrlPrefix($value, 'youtube.com/@', false), '@'));
            case 'github':
                return 'https://www.github.com/' . rawurlencode($this->_removeUrlPrefix($value, 'github.com'));
            case 'telegram':
                return 'https://t.me/' . rawurlencode($this->_removeUrlPrefix($value, 't.me'));
            case 'tiktok':
                return 'https://tiktok.com/@' . rawurlencode(ltrim($this->_removeUrlPrefix($value, 'tiktok.com/@', false), '@'));
            case 'tumblr':
            case 'linkedin':
            case 'flickr':
            case 'rss':
                return $value;
            case 'threads':
                return 'https://threads.net/@' . rawurlencode(ltrim($this->_removeUrlPrefix($value, 'threads.net/@', false), '@'));
            case 'bluesky':
                if (strpos($value, '@') === 0) {
                    $value = rawurlencode(substr($value, 1));
                }
                return 'https://bsky.app/profile/' . rawurlencode($this->_removeUrlPrefix($value, 'bsky.app/profile'));
            case 'mastodon':
                return 'https://mastodon.social/@' . rawurlencode(ltrim($this->_removeUrlPrefix($value, 'mastodon.social/@', false), '@'));
            default:
                return $this->_application->Filter('social_media_url', $value, [$name, $value]);
        }
    }

    protected function _removeUrlPrefix($url, $prefix, $removeSlash = true)
    {
        if ($removeSlash) $prefix = rtrim($prefix, '/') . '/';
        if (false !== $pos = strpos($url, $prefix)) {
            $url = substr($url, $pos + strlen($prefix));
        }
        return $url;
    }

    public function fieldGetTypeNames()
    {
        return ['social_accounts'];
    }

    public function fieldGetType($name)
    {
        return new FieldType\AccountsFieldType($this->_application, $name);
    }

    public function fieldGetWidgetNames()
    {
        return ['social_accounts'];
    }

    public function fieldGetWidget($name)
    {
        return new FieldWidget\AccountsFieldWidget($this->_application, $name);
    }
    
    public function fieldGetFilterNames()
    {
        return ['social_accounts'];
    }

    public function fieldGetFilter($name)
    {
        return new FieldFilter\AccountsFieldFilter($this->_application, $name);
    }
    
    public function fieldGetRendererNames()
    {
        return ['social_accounts', 'social_twitter_feed', 'social_facebook_page', 'social_facebook_messenger_link'];
    }

    public function fieldGetRenderer($name)
    {
        switch ($name) {
            case 'social_accounts':
                return new FieldRenderer\AccountsFieldRenderer($this->_application, $name);
            case 'social_twitter_feed':
                return new FieldRenderer\TwitterFeedFieldRenderer($this->_application, $name);
            case 'social_facebook_page':
                return new FieldRenderer\FacebookPageFieldRenderer($this->_application, $name);
            case 'social_facebook_messenger_link':
                return new FieldRenderer\FacebookMessengerLinkFieldRenderer($this->_application, $name);
        }
    }
}
