<?php

namespace Ademti\WoocommerceMsrp\Admin;

use WC_Product;
use function add_action;
use function dirname;
use function esc_html;
use function sanitize_text_field;
use function wc_format_decimal;
use function wc_get_product;
use function wp_unslash;

class QuickEdit {

	/**
	 * @return void
	 */
	public function run() {
		// Show MSRP fields in the quick edit section of the product listing page.
		add_action( 'woocommerce_product_quick_edit_end', [ $this, 'quick_edit_fields' ] );

		// Output the product data to populate the quick edit fields for products.
		add_action( 'add_inline_data', [ $this, 'add_inline_data' ], 10, 2 );

		// Save the data when the quick edit form is submitted.
		add_action( 'woocommerce_product_quick_edit_save', [ $this, 'save_quick_edit_form' ] );
	}

	/**
	 * Outputs the field template for the quick edit form for MSRP prices.
	 *
	 * @return void
	 */
	public function quick_edit_fields() {
		require dirname( __DIR__, 2 ) . '/templates/quick_edit_show_fields.php';
	}

	/**
	 * Outputs the hidden price data used to populate the quick edit form when shown.
	 *
	 * @param $post
	 * @param $post_type
	 *
	 * @return void
	 */
	public function add_inline_data( $post, $post_type ) {
		if ( $post_type->name !== 'product' ) {
			return;
		}
		$product    = wc_get_product( $post->ID );
		$msrp_price = $product->get_meta( '_msrp_price', true );
		echo '<div class="_msrp_price">' . esc_html( $msrp_price ) . '</div>';
	}

	/**
	 * Saves the data from the quick edit form.
	 *
	 * @param $product
	 *
	 * @return void
	 */
	public function save_quick_edit_form( WC_Product $product ) {
		$msrp_price = wc_format_decimal(
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
			sanitize_text_field( wp_unslash( $_POST['_msrp_price'] ?? '' ) )
		);
		if ( $msrp_price === '' ) {
			$product->delete_meta_data( '_msrp_price' );
		} else {
			$product->update_meta_data( '_msrp_price', $msrp_price );
		}
	}
}
