<?php

namespace Ademti\WoocommerceMsrp\RestHandlers;

use WC_Product;

abstract class AbstractRestHandler {

	/**
	 * Register the schema hook.
	 *
	 * Applies to both REST handlers.
	 */
	public function run(): void {
		add_filter( 'woocommerce_rest_product_schema', [ $this, 'rest_api_product_schema' ], 99 );
	}

	/**
	 * Include a description of the msrp_price element in the REST schema.
	 *
	 * Applies to both REST handlers.
	 */
	public function rest_api_product_schema( array $schema ): array {
		// Add the MSRP price for products.
		$schema['msrp_price'] = [
			'description' => 'The MSRP price for the product.',
			'type'        => 'string',
			'context'     => [
				'view',
				'edit',
			],
		];
		// Cater for the bulk product endpoint.
		if ( isset( $schema['variations'] ) &&
			'object' === $schema['variations']['items']['type'] ) {
			$schema['variations']['items']['properties']['msrp_price'] = [
				'description' => 'The MSRP price for the variation.',
				'type'        => 'string',
				'context'     => [
					'view',
					'edit',
				],
			];
		}

		return $schema;
	}

	/**
	 * Get the MSRP for a non-variable product
	 *
	 * @param  WC_Product  $current_product  The product the MSRP is required for
	 *
	 * @return string                  The MSRP, or empty string
	 */
	protected function get_msrp_for_single_product( WC_Product $current_product ): string {
		return $current_product->get_meta( '_msrp_price' );
	}

	/**
	 * Get the MSRP for a single variation.
	 *
	 * @param  WC_Product  $current_product  The product the MSRP is required for
	 *
	 * @return string                   The MSRP, or empty string
	 */
	protected function get_msrp_for_variation( WC_Product $current_product ): string {
		return $current_product->get_meta( '_msrp' );
	}
}
