<?php

namespace Ademti\WoocommerceMsrp\RestHandlers;

use WC_Product;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Handles the MSRP price on products in the WooCommerce v2 REST API.
 */
class RestHandler extends AbstractRestHandler {

	/**
	 * Run the handler.
	 *
	 * @return void
	 */
	public function run(): void {
		// General REST support.
		parent::run();
		// REST API support for WooCommerce post-v2.6.
		add_filter( 'woocommerce_rest_prepare_product_object', [ $this, 'rest_api_price_output_v2_simple' ], 10, 2 );
		add_filter(
			'woocommerce_rest_prepare_product_variation_object',
			[ $this, 'rest_api_price_output_v2_variation' ],
			10,
			2
		);
		add_filter( 'woocommerce_rest_pre_insert_product_object', [ $this, 'rest_api_maybe_update_msrp_v2' ], 10, 2 );
		add_filter(
			'woocommerce_rest_pre_insert_product_variation_object',
			[ $this, 'rest_api_maybe_update_msrp_v2' ],
			10,
			3
		);
	}

	/**
	 * Include MSRP prices in REST API for products/xx
	 */
	public function rest_api_price_output_v2_simple(
		WP_REST_Response $response,
		WC_Product $product
	): WP_REST_Response {
		// Remove MSRP entries from the meta_data element.
		if ( isset( $response->data['meta_data'] ) ) {
			foreach ( $response->data['meta_data'] as $key => $val ) {
				if ( in_array( $val->key, [ '_msrp', '_msrp_price' ], true ) ) {
					unset( $response->data['meta_data'][ $key ] );
				}
			}
			$response->data['meta_data'] = array_values( $response->data['meta_data'] );
		}

		// Do nothing else if we already have the data.
		if ( isset( $response->data['msrp_price'] ) ) {
			return $response;
		}

		// Add the MSRP price into the response.
		if ( 'variation' === $product->get_type() ) {
			$response->data['msrp_price'] = $this->get_msrp_for_variation( $product );
		} elseif ( 'variable' !== $product->get_type() ) {
			$response->data['msrp_price'] = $this->get_msrp_for_single_product( $product );
		}

		return $response;
	}

	/**
	 * Include MSRP prices in REST API for products/xx/variation/yy
	 */
	public function rest_api_price_output_v2_variation(
		WP_REST_Response $response,
		WC_Product $product
	): WP_REST_Response {
		// Remove MSRP entries from the meta_data element.
		foreach ( $response->data['meta_data'] as $key => $val ) {
			if ( in_array( $val->key, [ '_msrp', '_msrp_price' ], true ) ) {
				unset( $response->data['meta_data'][ $key ] );
			}
		}
		$response->data['meta_data'] = array_values( $response->data['meta_data'] );

		// Do nothing else if we already have the data.
		if ( isset( $response->data['msrp_price'] ) ) {
			return $response;
		}

		// Add the MSRP price into the response.
		$response->data['msrp_price'] = $this->get_msrp_for_variation( $product );

		return $response;
	}

	/**
	 * Update the MSRP for a product.
	 */
	public function rest_api_maybe_update_msrp_v2( WC_Product $product, WP_REST_Request $request ): WC_Product {
		if ( ! isset( $request['msrp_price'] ) ) {
			return $product;
		}
		if ( 'variation' === $product->get_type() ) {
			$key = '_msrp';
		} else {
			$key = '_msrp_price';
		}
		if ( empty( $request['msrp_price'] ) ) {
			$product->delete_meta_data( $key );
		} else {
			$product->update_meta_data( $key, $request['msrp_price'] );
		}

		$product->save();

		return $product;
	}
}
