#!/bin/sh

. /usr/lib/tuned/functions


function ver_comp () {
    if [[ $1 == $2 ]]
    then
        return 0
    fi
    local IFS=.
    local i ver1=($1) ver2=($2)
    # fill empty fields in ver1 with zeros
    for ((i=${#ver1[@]}; i<${#ver2[@]}; i++))
    do
        ver1[i]=0
    done
    for ((i=0; i<${#ver1[@]}; i++))
    do
        if [[ -z ${ver2[i]} ]]
        then
            # fill empty fields in ver2 with zeros
            ver2[i]=0
        fi
        if ((10#${ver1[i]} > 10#${ver2[i]}))
        then
            return 1
        fi
        if ((10#${ver1[i]} < 10#${ver2[i]}))
        then
            return 2
        fi
    done
    return 0
}

function lve_ver_comp () {
    local current_lve_ver=$(echo $1 | awk -F "lve" '{print $2}' | awk -F ".el" '{print $1}')
    local supported_lve_ver=$2
    ver_comp "${current_lve_ver}" "${supported_lve_ver}"
    if [[ $? == 0 || $? == 1 ]]; then
        return 0
    else
        return 1
    fi
}

start() {
    local dev_path
    local dev_name
    local rotational
    local elevator
    local vz_module
    if [ -e /proc/vz/vestat ]; then
        # CloudLinux profile settings aren't compatible with OpenVZ
        # set parameters for oom-killer
        if [ -e /proc/vz/oom_score_adj ]; then
            cat /etc/tuned-profiles-cloudlinux/oom-groups.conf > /proc/vz/oom_score_adj
        fi
        return 0
    fi
    lve_ver_comp "$(uname -r)" "1.4.50" # minimal supported version for `vfs_cache_min_ratio` kernel param (see. LU-578)
    if [ $? == 0 ] && [ -f /proc/sys/vm/vfs_cache_min_ratio ]; then
        echo 0 > /proc/sys/vm/vfs_cache_min_ratio
    fi
    # disable tcache/tswap OpenVZ subsystems
    for vz_module in tcache tswap; do
        save_set_sys "active_module_${vz_module}" "/sys/module/${vz_module}/parameters/active" 'N'
    done
    # block devices configuration
    for dev_path in ${DISKS_SYS}; do
        dev_name=$(eval LANG=C /bin/basename "${dev_path}" 2>/dev/null)
        rotational=$(eval LANG=C /bin/cat "${dev_path}/queue/rotational" 2>/dev/null)
        if [ "${rotational}" -eq 1 ]; then
            elevator='deadline'
        else
            elevator='noop'
        fi
        set_elevator "${dev_path}/queue/scheduler" ${elevator} 2>/dev/null
        save_set_sys "nr_requests_${dev_name}" "${dev_path}/queue/nr_requests" '1024' 2>/dev/null
    done
    # disable hugepages
    save_set_sys 'transparent_hugepage_khugepaged_defrag' '/sys/kernel/mm/transparent_hugepage/khugepaged/defrag' '0' 2>/dev/null
    save_set_sys 'transparent_hugepage_defrag' '/sys/kernel/mm/transparent_hugepage/defrag' 'never' 2>/dev/null
    save_set_sys 'transparent_hugepage_enabled' '/sys/kernel/mm/transparent_hugepage/enabled' 'never' 2>/dev/null
    return 0
}


stop() {
    local dev_path
    local dev_name
    local vz_module
    if [ -e /proc/vz/vestat ]; then
        # CloudLinux profile settings aren't compatible with OpenVZ
        return 0
    fi
    # revert tcache/tswap OpenVZ subsystems state
    for vz_module in tcache tswap; do
        restore_sys "active_module_${vz_module}" "/sys/module/${vz_module}/parameters/active"
    done
    # revert block devices configuration
    for dev_path in ${DISKS_SYS}; do
        dev_name=$(eval LANG=C /bin/basename "${dev_path}" 2>/dev/null)
        restore_elevator "${dev_path}/queue/scheduler" 2>/dev/null
        restore_sys "nr_requests_${dev_name}" "${dev_path}/queue/nr_requests" 2>/dev/null
    done
    # revert hugepages configuration
    restore_sys 'transparent_hugepage_khugepaged_defrag' '/sys/kernel/mm/transparent_hugepage/khugepaged/defrag' 2>/dev/null
    restore_sys 'transparent_hugepage_defrag' '/sys/kernel/mm/transparent_hugepage/defrag' 2>/dev/null
    restore_sys 'transparent_hugepage_enabled' '/sys/kernel/mm/transparent_hugepage/enabled' 2>/dev/null
    return 0
}


process $@
